#include "VBOData.h"

CVBOData::CVBOData(GLuint pProgram) :
	GLSLCount(0) {
	glGenBuffers(1, &id);
	program = pProgram;
}

CVBOData::~CVBOData() {
	Unbind();
	GLSLLocation.clear();
	GLSLName.clear();

	glDeleteBuffers(1,&id);
}

void CVBOData::DrawBuffer() const {
}

GLfloat * CVBOData::StartBufferOperation(MAP type) {
	return (GLfloat*) glMapBufferARB(target, type);
}

void CVBOData::StopBufferOperation() {
	glUnmapBufferARB(target);
}

int CVBOData::GetPrimitiveCount(void) const {
	return size;
}

void CVBOData::Bind(void) const {
	glBindBuffer(target, id);
}
void CVBOData::Unbind(void) const {
	glBindBuffer(target, 0);
}

/*
 *
 *    Feedback Buffer
 *
 */

CFeedbackBuffer::CFeedbackBuffer(GLuint pProgram, GLsizei sz, GLenum wPrimitive) :
	CVBOData(pProgram), usage(GL_STATIC_DRAW_ARB), data(NULL),
			bufferMode(GL_INTERLEAVED_ATTRIBS_NV), writePrimitive(wPrimitive),
			feedbackInterleavedFormat(GL_C4F_N3F_V3F), locked(NONE) {
	glGenQueries(1, &primitiveQuery);
	target = GL_ARRAY_BUFFER;
	size = sz;
}

CFeedbackBuffer::~CFeedbackBuffer() {
	if (glIsQuery(primitiveQuery))
		glDeleteQueries(1,&primitiveQuery);
}

void CFeedbackBuffer::Install() {
	Bind();

	glBufferData(target, size, data ,usage);

	GLstateTest();

	//TODO: make this two next methods more generic
	glTransformFeedbackVaryingsNV(program, GLSLCount, &GLSLLocation[0], bufferMode);
	glBindBufferOffsetNV(GL_TRANSFORM_FEEDBACK_BUFFER_NV, 0, id, 0);

	Unbind();
}

void CFeedbackBuffer::AddGLSLVarying(std::string name) {
	Bind();
	GLSLName.push_back(name);
	glActiveVaryingNV(program, GLSLName[GLSLCount].c_str());
	GLSLLocation.push_back(glGetVaryingLocationNV(program, GLSLName[GLSLCount].c_str()));
	GLSLCount++;
	Unbind();
}

int CFeedbackBuffer::GetPrimitiveCount(void) const {
	GLuint primitives_written = 0;
	glGetQueryObjectuiv(primitiveQuery, GL_QUERY_RESULT, &primitives_written);
	return primitives_written;
}

int CFeedbackBuffer::GetVertexCount(void) const {
	switch (writePrimitive) {
	case GL_POINTS:
		return GetPrimitiveCount();
		break;
	case GL_TRIANGLES:
		return GetPrimitiveCount()*3;
		break;

	case GL_LINES:
		return GetPrimitiveCount()*2;
		break;
	}

	MarkError();

	return 0;
}

unsigned int CFeedbackBuffer::GetElement(int n) const {
	switch (feedbackInterleavedFormat) {
	case GL_C4F_N3F_V3F:
		return n;
		break;
	}

	MarkError();

	return 0;
}

const float* CFeedbackBuffer::GetVertex(int n) {
	n*=10;
	if (locked == NONE)
		memoryPointerRead = StartBufferOperation(READ);

	if (n > GetPrimitiveCount()*3*10) {
		MarkError();
		return NULL;
	}
	return &memoryPointerRead[n+4+3];
}

void CFeedbackBuffer::AddColor(int n, float r, float g, float b) {
	if ((locked == NONE)||(locked == READ)) {
		StopBufferOperation();
		memoryPointer = StartBufferOperation(READ_WRITE);
	}

	memoryPointer[(GetElement(n))*10] = r;
	memoryPointer[(GetElement(n))*10+1] = g;
	memoryPointer[(GetElement(n))*10+2] = b;
}

void CFeedbackBuffer::SetColor(int n, float r, float g, float b) {
	AddColor(n, r, g, b);
}

GLfloat * CFeedbackBuffer::StartBufferOperation(MAP type) {
	Bind();
	locked = type;
	return (GLfloat*) glMapBufferARB(GL_TRANSFORM_FEEDBACK_BUFFER_NV, type);
}

void CFeedbackBuffer::StopBufferOperation() {
	locked = NONE;
	GLboolean stopSuccess= glUnmapBufferARB(GL_TRANSFORM_FEEDBACK_BUFFER_NV);
	Unbind();
	if (!stopSuccess)
		std::cerr << "Error Unmaping buffer";
	GLstateTest();
}

void CFeedbackBuffer::BeginTransformFeedback() {
	CheckWriteStatus();
	Bind();
	glEnable(GL_RASTERIZER_DISCARD_NV); // disable rasterization
	glBeginTransformFeedbackNV(writePrimitive);
	glBeginQuery(GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN_NV, primitiveQuery);
	Unbind();
}

void CFeedbackBuffer::EndTransformFeedback() {
	CheckWriteStatus();
	Bind();
	glEndQuery(GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN_NV);
	glEndTransformFeedbackNV();
	glDisable(GL_RASTERIZER_DISCARD_NV);
	Unbind();
}

void CFeedbackBuffer::CheckWriteStatus(void) {
	if (locked)
		StopBufferOperation();
}

void CFeedbackBuffer::DrawArray(GLenum drawPrimitive) const {
	glInterleavedArrays(feedbackInterleavedFormat, 0, NULL);

	switch (drawPrimitive) {
	case GL_POINTS:
		glDrawArrays(drawPrimitive, 0, GetPrimitiveCount());
		break;
	case GL_TRIANGLES:
		glDrawArrays(drawPrimitive, 0, 3*GetPrimitiveCount());
		break;
	case GL_LINES:
		glDrawArrays(drawPrimitive, 0, 2*GetPrimitiveCount());
		break;
	}

}

bool CFeedbackBuffer::Draw(void) {
	CheckWriteStatus();
	Bind();
	glPushClientAttrib(GL_CLIENT_VERTEX_ARRAY_BIT);
	DrawArray(writePrimitive);
	glPopClientAttrib();
	Unbind();
	return true;
}

